<?php
/**
 * Zikula Application Framework
 * @copyright (c) 2001, Zikula Development Team
 * @link http://www.zikula.org
 * @version $Id: pninit.php 25856 2009-06-05 10:44:27Z Guite $
 * @license GNU/GPL - http://www.gnu.org/copyleft/gpl.html
 * @package Zikula_System_Modules
 * @subpackage Blocks
 */

/**
 * initialise the blocks module
 *
 * This function is only ever called once during the lifetime of a particular
 * module instance.
 * This function MUST exist in the pninit file for a module
 *
 * @author       Mark West
 * @return       bool       true on success, false otherwise
 */
function blocks_init()
{
    // create blocks table
    // appropriate error message and return
    if (!DBUtil::createTable('blocks')) {
        return false;
    }

    // create userblocks table
    if (!DBUtil::createTable('userblocks')) {
        return false;
    }

    // create block positions table
    if (!DBUtil::changeTable('block_positions')) {
        return false;
    }

    // create block placements table
    if (!DBUtil::changeTable('block_placements')) {
        return false;
    }

    // Set a default value for a module variable
    pnModSetVar('Blocks', 'collapseable', 0);

    // Initialisation successful
    return true;
}

/**
 * upgrade the blocks module from an old version
 *
 * This function can be called multiple times
 * This function MUST exist in the pninit file for a module
 *
 * @author       Mark West
 * @return       bool       true on success, false otherwise
 */
function blocks_upgrade($oldversion)
{
    // update blocks table
    if (!DBUtil::changeTable('blocks')) {
        return false;
    }
    // update userblocks table
    if (!DBUtil::changeTable('userblocks')) {
        return false;
    }
    // update block positions table
    if (!DBUtil::changeTable('block_positions')) {
        return false;
    }
    // update block placements table
    if (!DBUtil::changeTable('block_placements')) {
        return false;
    }


    // Upgrade dependent on old version number
    switch($oldversion) {
        // version 2.2 was supplied with all .76x versions of PN
        case 2.2: // upgrade from 76x
            // create the default block positions
            // force load the API since it's not available until the module is activated
            pnModAPILoad('Blocks', 'admin', true);
            pnModAPIFunc('Blocks', 'admin', 'createposition', array('name' => 'left', 'description' => _BLOCKS_LEFTBLOCKS));
            pnModAPIFunc('Blocks', 'admin', 'createposition', array('name' => 'right', 'description' => _BLOCKS_RIGHTBLOCKS));
            pnModAPIFunc('Blocks', 'admin', 'createposition', array('name' => 'center', 'description' => _BLOCKS_CENTERBLOCKS));

            $prefix = pnDBGetTablePrefix('blocks');
            $fields = array($prefix.'_position',
                            $prefix.'_weight');
            DBUtil::dropColumn('blocks', $fields);
            DBUtil::dropIndex('bid_idx', 'blocks');
            break;
    }

    // Update successful
    return true;
}

/**
 * delete the blocks module
 *
 * This function is only ever called once during the lifetime of a particular
 * module instance
 * This function MUST exist in the pninit file for a module
 *
 * Since the blocks module should never be deleted we'all always return false here
 * @author       Mark West
 * @return       bool       false
 */
function blocks_delete()
{
    // Deletion not allowed
    return false;
}


/**
 * Add default block data for new installs
 * This is called after a complete pn installation since the blocks
 * need to be populated with module id's which are only available
 * once the install has been completed
 *
 */
function blocks_defaultdata()
{
    // create the default block positions - left, right and center for the traditional 3 column layout
    pnModAPILoad('Blocks', 'admin', true);
    // load language file
    pnModLangLoad('Blocks', 'init');

    // sanity check - truncate existing tables to ensure a clean blocks setup
    DBUtil::truncateTable('blocks');
    DBUtil::truncateTable('block_positions');
    DBUtil::truncateTable('block_placements');

    //pnModLoad('Blocks', 'admin', true);
    $left   = pnModAPIFunc('Blocks', 'admin', 'createposition', array('name' => 'left', 'description' => _BLOCKS_LEFTBLOCKS));
    $right  = pnModAPIFunc('Blocks', 'admin', 'createposition', array('name' => 'right', 'description' => _BLOCKS_RIGHTBLOCKS));
    $center = pnModAPIFunc('Blocks', 'admin', 'createposition', array('name' => 'center', 'description' => _BLOCKS_CENTERBLOCKS));

    // define an array of the default blocks
    $blocks = array();
    // build the menu content
    $lang = pnUserGetLang();
    $menucontent = array();
    $menucontent['displaymodules'] = '1';
    $menucontent['stylesheet']     = 'extmenu.css';
    $menucontent['template']       = 'blocks_block_extmenu.htm';
    $menucontent['blocktitles']    = array("$lang" => _BLOCKS_MENU_TITLE);
    // insert the links
    $menucontent['links'][$lang][] = array('name'   => _BLOCKS_MENU_1LINK,
                                           'url'    => '{homepage}',
                                           'title'  => _BLOCKS_MENU_1TITLE,
                                           'level'  => 0,
                                           'parent' => 0,
                                           'image'  => '',
                                           'active' => '1');
    $menucontent['links'][$lang][] = array('name'   => _BLOCKS_MENU_3LINK,
                                           'url'    => '{Admin:adminpanel:admin}',
                                           'title'  => _BLOCKS_MENU_3TITLE,
                                           'level'  => 0,
                                           'parent' => 0,
                                           'image'  => '',
                                           'active' => '1');
    $menucontent['links'][$lang][] = array('name'   => _BLOCKS_MENU_4LINK,
                                           'url'    => '{Users:logout}',
                                           'title'  => _BLOCKS_MENU_4TITLE,
                                           'level'  => 0,
                                           'parent' => 0,
                                           'image'  => '',
                                           'active' => '1');
    $menucontent = serialize($menucontent);
    $blocks[] = array('bkey' => 'extmenu', 'collapsable' => 1, 'defaultstate' => 1, 'language' => '',
                      'mid' => pnModGetIDFromName('Blocks'), 'title' => _BLOCKS_MENU_TITLE, 'content' => $menucontent, 'positions' => array($left));
    $blocks[] = array('bkey' => 'thelang', 'collapsable' => 1, 'defaultstate' => 1, 'language' => '',
                      'mid' => pnModGetIDFromName('Blocks'), 'title' => _BLOCKS_LANG_TITLE, 'content' => '', 'positions' => array($left));
    $blocks[] = array('bkey' => 'login', 'collapsable' => 1, 'defaultstate' => 1, 'language' => '',
                      'mid' => pnModGetIDFromName('Users'), 'title' => _BLOCKS_LOGIN_TITLE, 'positions' => array($right));
    $blocks[] = array('bkey' => 'online', 'collapsable' => 1, 'defaultstate' => 1, 'language' => '',
                      'mid' => pnModGetIDFromName('Users'), 'title' => _BLOCKS_ONLINE_TITLE, 'positions' => array($right));
    $blocks[] = array('bkey' => 'messages', 'collapsable' => 1, 'defaultstate' => 1, 'language' => '',
                      'mid' => pnModGetIDFromName('Admin_Messages'), 'title' => _BLOCKS_MESSAGES_TITLE, 'positions' => array($center));

    // create each block and then update the block
    // the create creates the initiial block record, the update sets the block placments
    foreach ($blocks as $position => $block) {
        $block['bid'] = pnModAPIFunc('Blocks', 'admin', 'create', $block);
        pnModAPIFunc('Blocks', 'admin', 'update', $block);
    }
    return;
}


/**
 * Upgrade for blocks tables data from 0.76x to 0.8+
 *
 * This function is called prior to the system upgrade
 */
function blocks_upgradefrom76_pre()
{
    // we need to get the blocks table prior to the update otherwise the position field get dropped
    // and we lose the data needed to rewrite any block positions

    // get site prefix - the old way....
    $prefix = pnConfigGetVar('prefix');

    // get the values from the blocks table
    $sql = "SELECT pn_bid, pn_position, pn_weight FROM {$prefix}_blocks";
    $result = DBUtil::executeSQL($sql);

    // format a result array in the correct format for the block placements table
    for (; !$result->EOF; $result->MoveNext()) {
        list($bid, $position, $weight) = $result->fields;
        // manually map the position id
        switch ($position) {
            case 'l': $pid = 1; break;
            case 'r': $pid = 2; break;
            case 'c': $pid = 3; break;
        }
        // only add blocks that are in one of the core positions
        if (isset($pid)) {
            $_SESSION['_PNUpgrader']['blocks'][] = array('pid' => $pid,
                                                         'bid' => $bid,
                                                         'order' => (int)$weight);
            unset($pid);														 
        }
    }
    $result->Close();
}

/**
 * Upgrade for blocks tables data from 0.76x to 0.8+
 *
 * This function is called after the system upgrade is complete
 */
function blocks_upgradefrom76_post()
{
    // now insert the backed up block positions into the placements table
    DBUtil::insertObjectArray($_SESSION['_PNUpgrader']['blocks'], 'block_placements');

    // get site prefix - the old way....
    $prefix = pnConfigGetVar('prefix');

    // fix some block module id's
    $sqls = array();
    $blocksmodid = pnModGetIDFromName('Blocks');
    if ($blocksmodid) {
        $sqls[] = "UPDATE {$prefix}_blocks SET pn_mid = {$blocksmodid}  WHERE pn_mid = 0 AND pn_bkey IN ('html','finclude','menu','text','thelang')";
    }
    $usersmodid = pnModGetIDFromName('Users');
    if ($usersmodid) {
        $sqls[] = "UPDATE {$prefix}_blocks SET pn_mid = {$usersmodid}   WHERE pn_mid = 0 AND pn_bkey IN ('login','online','user')";
    }
    $pollsmodid = pnModGetIDFromName('Polls');
    if ($pollsmodid) {
        $sqls[] = "UPDATE {$prefix}_blocks SET pn_mid = {$pollsmodid}   WHERE pn_mid = 0 AND pn_bkey IN ('poll')";
    }
    $bannersmodid = pnModGetIDFromName('Banners');
    if ($bannersmodid) {
        $sqls[] = "UPDATE {$prefix}_blocks SET pn_mid = {$bannersmodid} WHERE pn_mid = 0 AND pn_bkey IN ('banners')";
    }
    $newsmodid = pnModGetIDFromName('News');
    if ($newsmodid) {
        $sqls[] = "UPDATE {$prefix}_blocks SET pn_mid = {$newsmodid}    WHERE pn_mid = 0 AND pn_bkey IN ('big','past','stories')";
    }
    $searchmodid = pnModGetIDFromName('Search');
    if ($searchmodid) {
        $sqls[] = "UPDATE {$prefix}_blocks SET pn_mid = {$searchmodid}  WHERE pn_mid = 0 AND pn_bkey IN ('search')";
    }

    foreach ($sqls as $sql) {
        DBUtil::executeSQL($sql);
    }
}
