<?php
require_once 'Zend/View/Interface.php'; 
 
/**
 * Adapter class
 */ 
class Addax_View_Adapter_Smarty implements Zend_View_Interface 
{ 
    /**
     * Default value of directories and files
     */ 
    const CLASS_FILE_NAME      = 'Smarty.class.php'; 
    const DEFAULT_TEMPLATE_DIR = 'templates'; 
    const DEFAULT_COMPILE_DIR  = 'templates_c'; 
    const DEFAULT_CONFIG_DIR   = 'configs'; 
    const DEFAULT_CACHE_DIR    = 'cache'; 
 
    /**
     * Error Messages
     */ 
    const INVALID_DIR    = 'Invalid directory specified.'; 
    const INVALID_NAME   = 'Invalid name specified.'; 
    const UNWRITABLE_DIR = 'Unwritable directory specified.'; 
    const NOUSES_CACHE   = 'No uses cache.'; 
 
    /**
     * Smarty instance
     *
     * @var Smarty
     */ 
    protected $_smarty; 
 
    /**
     * Cache mode
     *
     * @var integer 0,1 or 2
     */ 
    protected $_cacheMode; 
 
    /**
     * work path
     *
     * @var string
     */ 
    protected $_workPath; 
 
    /**
     * ---------------------------------------
     * At first, implement methods extended from Zend_View_Interface
     * ---------------------------------------
     */ 
 
    /**
     * Instatiation
     *
     * @param string $workPath Template, compiled template and cache
     * directory exist under the directory if specified.
     * The directories also can be specified with setWorkPath() or
     * setScriptPath() after instanciation.
     * If you want to customize position of directories, you should use
     * setTemplatePath, setCompilePath and setCachePath instead of
     * specifing with the instanciation or setWorkPath or setScriptPath.
     * @param string $smartyDir absolute directory that the Smarty is
     * installed. It is required if the Smarty class does not exist on
     * the include_path.
     * @param integer $cacheMode 0=no cache, 1=cache all template,
     * 2=cache each template
     *
     * @param array $optparam option parameters.
     *
     * @return void
     * @throws Addax_View_Adapter_Exception
     * @see http://www.smarty.net/
     */ 
    public function __construct($workPath = null, $smartyDir = null, $cacheMode = 0, $optparam = array()) 
    { 
        if ($smartyDir !== null) 
        { 
            $smartyClassPath = $smartyDir . DIRECTORY_SEPARATOR . self::CLASS_FILE_NAME; 
            if (is_file($smartyClassPath) === false) 
            { 
                require_once 'Addax/View/Adapter/Exception.php'; 
                throw new Addax_View_Adapter_Exception('Smarty is not installed.'); 
            } 
        } else { 
            $smartyClassPath = self::CLASS_FILE_NAME; 
        }
        require_once $smartyClassPath; 
        $this->_smarty = new Smarty; 
 
        if ($cacheMode !== 0) 
        { 
            $this->setCacheMode($cacheMode); 
        } 
 
        if ($workPath !== null) 
        { 
            $this->setWorkPath($workPath); 
        } 
 
        if ($optparam !== array()) 
        { 
            $this->setProperties($optparam); 
        } 
    } 
 
    /**
     * Return the Smarty instance
     *
     * @return Smarty
     */ 
    public function getEngine() 
    { 
        return $this->_smarty; 
    } 
 
    /**
     * Set parent of the path to work path.
     * As a result, set the path to template path.
     *
     * @param string $path template path.
     * @return void
     * @throws Addax_View_Adapter_Exception
     */ 
    public function setScriptPath($path) 
    { 
        if ($this->_workPath !== null) 
        { 
            //Script path and work path already set. 
            return; 
        } 
 
        $workPath = dirname($path); 
        if ($workPath === '.') 
        { 
            require_once 'Addax/View/Adapter/Exception.php'; 
            throw new Addax_View_Adapter_Exception(self::INVALID_DIR); 
        } 
 
        $this->setWorkPath($workPath); 
    } 
 
    /**
     * Retrieve the current template path
     *
     * @return array
     */ 
    public function getScriptPaths() 
    { 
        return array($this->getTemplatePath()); 
    } 
 
    /**
     * Set parent of the path to work path.
     * As a result, set the path to template path.
     *
     * @param string $path
     * @param string $prefix
     * @return void
     */ 
    public function setBasePath($path, $prefix = 'Zend_View') 
    { 
        $this->setScriptPath($path); 
    } 
 
    /**
     * Add path
     *
     * @param string $path
     * @param string $prefix
     * @return void
     * @throws Addax_View_Adapter_Exception
     */ 
    public function addBasePath($path, $prefix = 'Zend_View') 
    { 
        require_once 'Addax/View/Adapter/Exception.php'; 
        throw new Addax_View_Adapter_Exception('Can not add path.'); 
    } 
 
    /**
     * Assign a variable to the template
     *
     * @param string $key variable name.
     * @param mixed $val variable value.
     * @return void
     */ 
    public function __set($key, $val) 
    { 
        $this->_smarty->assign_by_ref($key, $val); 
    } 
 
    /**
     * Allows testing with empty() and isset() to work
     *
     * @param string $key variable name.
     * @return boolean
     */ 
    public function __isset($key) 
    { 
        return (null !== $this->_smarty->get_template_vars($key)); 
    } 
 
    /**
     * Allows unset() on object properties to work
     *
     * @param string $key variable name.
     * @return void
     */ 
    public function __unset($key) 
    { 
        $this->_smarty->clear_assign($key); 
    } 
 
    /**
     * Assign variables to the template
     *
     * Allows setting a specific key to the specified value, OR passing an array
     * of key => value pairs to set en masse.
     *
     * @see __set()
     * @param string|array $spec The assignment strategy to use (key or array of key
     * => value pairs)
     * @param mixed $value (Optional) If assigning a named variable, use this
     * as the value.
     * @return void
     */ 
    public function assign($spec, $value = null) 
    { 
        if (is_array($spec)) 
        { 
            $this->_smarty->assign_by_ref($spec); 
            return; 
        } 
 
 
        $this->_smarty->assign_by_ref($spec, $value); 
    } 
 
	public function assignOld($spec, $value = null)
    {
        if (is_string($spec))
        {
            // assign by name and value
            if ('_' == substr($spec, 0, 1)) {
                require_once 'Zend/View/Exception.php';
                throw new Zend_View_Exception('Setting private or protected class members is not allowed');
            }
            
            $this->_smarty->assign($spec, $value);
        }
        elseif (is_array($spec))
        {
			// assign from associative array
            $error = false;
            foreach ($spec as $key => $value)
            {
				if ('_' == substr($key, 0, 1)) 
				{
                    $error = true;
                    break;
                }
                $this->_smarty->assign($key, $value);
            }
			if ($error) {
                require_once 'Zend/View/Exception.php';
                throw new Zend_View_Exception('Setting private or protected class members is not allowed');
            }
        }
        else
        {
			require_once 'Zend/View/Exception.php';
            throw new Zend_View_Exception('assign() expects a string or array, got '.gettype($spec));
        }
    }
	
	
	public function assignOld($spec, $value = null)
    {
        if (is_string($spec))
        {
            // assign by name and value
            if ('_' == substr($spec, 0, 1)) {
                require_once 'Zend/View/Exception.php';
                throw new Zend_View_Exception('Setting private or protected class members is not allowed');
            }
            
            $this->_smarty->assign($spec, $value);
        }
        elseif (is_array($spec))
        {
			// assign from associative array
            $error = false;
            foreach ($spec as $key => $value)
            {
				if ('_' == substr($key, 0, 1)) 
				{
                    $error = true;
                    break;
                }
                $this->_smarty->assign($key, $value);
            }
			if ($error) {
                require_once 'Zend/View/Exception.php';
                throw new Zend_View_Exception('Setting private or protected class members is not allowed');
            }
        }
        else
        {
			require_once 'Zend/View/Exception.php';
            throw new Zend_View_Exception('assign() expects a string or array, got '.gettype($spec));
        }
    }
    /**
     * Clear all assigned variables
     *
     * @return void
     */ 
    public function clearVars() 
    { 
        $this->_smarty->clear_all_assign(); 
    } 
 
    /**
     * Processes a template and display.
     *
     * @see output($name)
     * @param string $name template to process.
     * @return void.
     */ 
    public function render($name) 
    { 
        $this->_smarty->display($name); 
    } 
 
    /**
     * ---------------------------------------
     * At seconde, implement methods as like Zend_View_Abstract
     * ---------------------------------------
     */ 
 
    /**
     * Retrieve a variable from the template
     *
     * @param string $key variable name.
     * @return mixed $val variable value.
     */ 
    public function &__get($key) 
    { 
        return $this->_smarty->get_template_vars($key); 
    } 
 
    /**
     * Retrieve variables from the template
     *
     * @return array
     */ 
    public function getVars() 
    { 
        return $this->_smarty->get_template_vars(); 
    } 
 
    /**
     * ---------------------------------------
     * At last, implement own methods of this class
     * ---------------------------------------
     */ 
 
    /**
     * Set work path
     *
     * @param string $path work path. Directories exist
     *  under the path.
     * @return void
     * @throws Addax_View_Adapter_Exception
     */ 
    public function setWorkPath($path) 
    { 
        if (is_dir($path) === false) 
        { 
            require_once 'Addax/View/Adapter/Exception.php'; 
            throw new Addax_View_Adapter_Exception(self::INVALID_DIR); 
        } 
        $this->setTemplatePath( 
                    $path . DIRECTORY_SEPARATOR . self::DEFAULT_TEMPLATE_DIR); 
        $this->setCompilePath( 
                    $path . DIRECTORY_SEPARATOR . self::DEFAULT_COMPILE_DIR); 
 
        if ($this->_cacheMode !== 0) 
        { 
            $this->setCachePath( 
                    $path . DIRECTORY_SEPARATOR . self::DEFAULT_CACHE_DIR); 
        } 
 
        $this->_workPath = $path; 
    } 
 
    /**
     * Retrieve the work path
     *
     * @return string
     */ 
    public function getWorkPath() 
    { 
        return $this->_workPath; 
    } 
 
    /**
     * Processes a template and returns the output string.
     *
     * @see render($name)
     * @param string $name template to process.
     * @return string
     */ 
    public function output($name) 
    { 
        return $this->_smarty->fetch($name); 
    } 
 
    /**
     * Retrieve the current template path
     *
     * @return string
     */ 
    public function getScriptPath() 
    { 
        return $this->getTemplatePath(); 
    } 
 
    /**
     * Set the path to the template path
     *
     * @param string $path template path.
     * @return void
     * @throws Addax_View_Adapter_Exception
     */ 
    public function setTemplatePath($path) 
    { 
        if (is_dir($path) === false) 
        { 
            require_once 'Addax/View/Adapter/Exception.php'; 
            throw new Addax_View_Adapter_Exception(self::INVALID_DIR); 
        } 
 
        $this->_smarty->template_dir = $path; 
    } 
 
    /**
     * Retrieve the current template path
     *
     * @return string
     */ 
    public function getTemplatePath() 
    { 
        return $this->_smarty->template_dir; 
    } 
 
    /**
     * Set the path to the compiled template path
     *
     * @param string $path compiled template path.
     * @return void
     * @throws Addax_View_Adapter_Exception
     */ 
    public function setCompilePath($path) 
    { 
        if (is_dir($path) === false) 
        { 
            require_once 'Addax/View/Adapter/Exception.php'; 
            throw new Addax_View_Adapter_Exception(self::INVALID_DIR); 
        } 
 
        if (is_writable($path) === false) 
        { 
            require_once 'Addax/View/Adapter/Exception.php'; 
            throw new Addax_View_Adapter_Exception(self::UNWRITABLE_DIR); 
        } 
 
        $this->_smarty->compile_dir = $path; 
    } 
 
    /**
     * Retrieve the current compiled template path
     *
     * @return string
     */ 
    public function getCompilePath() 
    { 
        return $this->_smarty->compile_dir; 
    } 
 
    /**
     * Set the path to the cache path
     *
     * @param string $path cache path.
     * @return void
     * @throws Addax_View_Adapter_Exception
     */ 
    public function setCachePath($path) 
    { 
        if (is_dir($path) === false) 
        { 
            require_once 'Addax/View/Adapter/Exception.php'; 
            throw new Addax_View_Adapter_Exception(self::INVALID_DIR); 
        } 
 
        if (is_writable($path) === false) 
        { 
            require_once 'Addax/View/Adapter/Exception.php'; 
            throw new Addax_View_Adapter_Exception(self::UNWRITABLE_DIR); 
        } 
 
        $this->_smarty->cache_dir = $path; 
    } 
 
    /**
     * Retrieve the current cache path
     *
     * @return string
     */ 
    public function getCachePath() 
    { 
        return $this->_smarty->cache_dir; 
    } 
 
    /**
     * Set the path to the configuration path
     *
     * @param string $path configuration path
     * @return void
     * @throws Addax_View_Adapter_Exception
     */ 
    public function setConfigPath($path) 
    { 
        if (is_dir($path) === false) 
        { 
            require_once 'Addax/View/Adapter/Exception.php'; 
            throw new Addax_View_Adapter_Exception(self::INVALID_DIR); 
        } 
 
        $this->_smarty->config_dir = $path; 
    } 
 
    /**
     * Retrieve the current configuration path
     *
     * @return string
     */ 
    public function getConfigPath() 
    { 
        return $this->_smarty->config_dir; 
    } 
 
    /**
     * Set the mode to the cache mode
     *
     * @param string $mode cache mode.
     * @return void
     * @throws Addax_View_Adapter_Exception
     */ 
    public function setCacheMode($mode) 
    { 
        if (in_array($mode, array(0, 1, 2)) === false) 
        { 
            require_once 'Addax/View/Adapter/Exception.php'; 
            throw new Addax_View_Adapter_Exception('Invalid cache mode specified.'); 
        } else { 
            $this->_cacheMode = $mode; 
            $this->_smarty->caching = $mode; 
        } 
    } 
 
    /**
     * Retrieve the cache mode
     *
     * @return string
     * @throws Addax_View_Adapter_Exception
     */ 
    public function getCacheMode() 
    { 
        if ($this->_smarty->caching !== $this->_cacheMode) 
        { 
            require_once 'Addax/View/Adapter/Exception.php'; 
            throw new Addax_View_Adapter_Exception('Inconsistent cache mode.'); 
        } 
 
        return $this->_cacheMode; 
    } 
 
    /**
     * Set the properties of Smarty
     *
     * @param array $properties properties of Smarty (name => value)
     * @return void
     */ 
    public function setProperties($properties = array()) 
    { 
        foreach ($properties as $key => $value) 
        { 
            $this->setProperty($key, $value); 
        } 
    } 
 
    /**
     * Set the value to property of Smarty
     *
     * @param string $key property name
     * @param mixed $value property value
     * @return void
     * @throws Addax_View_Adapter_Exception
     */ 
    public function setProperty($key, $value) 
    { 
        if (!property_exists($this->_smarty, $key)) 
        { 
            require_once 'Addax/View/Adapter/Exception.php'; 
            throw new Addax_View_Adapter_Exception(self::INVALID_NAME); 
        } 
 
        switch ($key) 
        { 
            case 'template_dir': 
                $this->setTemplatePath($value); 
                break; 
            case 'compile_dir': 
                $this->setCompilePath($value); 
                break; 
            case 'cache_dir': 
                $this->setCachePath($value); 
                break; 
            case 'config_dir': 
                $this->setConfigPath($value); 
                break; 
            case 'caching': 
                if ($this->_cacheMode !== null) 
                { 
                    require_once 'Addax/View/Adapter/Exception.php'; 
                    throw new Addax_View_Adapter_Exception('Cache mode already specified.'); 
                } 
                $this->setCacheMode($value); 
                break; 
            default: 
                if (is_array($this->_smarty->$key)) 
                { 
                    $this->_smarty->{$key}[] = $value; 
                } else { 
                    $this->_smarty->$key = $value; 
                } 
                break; 
        } 
    } 
 
    /**
     * Retrieve the properties of Smarty
     *
     * @return array
     */ 
    public function getProperties() 
    { 
        $ret = array(); 
 
        foreach ($this->_smarty as $key => $value) 
        { 
            if($key{0} !== '_') 
            { 
                $ret[$key] = $value; 
            } 
        } 
 
        return $ret; 
    } 
 
    /**
     * Retrieve the property of Smarty
     *
     * @param string $key property name
     * @return string|array
     * @throws Addax_View_Adapter_Exception
     */ 
    public function getProperty($key) 
    { 
        if(property_exists($this->_smarty, $key) === false) 
        { 
            require_once 'Addax/View/Adapter/Exception.php'; 
            throw new Addax_View_Adapter_Exception(self::INVALID_NAME); 
        } 
        return $this->_smarty->$key; 
    } 
 
    /**
     * Retrieve whether or not the template is cached
     *
     * @param string $name template name
     * @return string|false
     */ 
    public function isCached($name) 
    { 
        if ($this->_cacheMode === 0) 
        { 
            require_once 'Addax/View/Adapter/Exception.php'; 
            throw new Addax_View_Adapter_Exception(self::NOUSES_CACHE); 
        } 
        return $this->_smarty->is_cached($name); 
    } 
 
    /**
     * Clear compiled template
     *
     * @param string $name template name
     * @param string $time expiration time (second)
     * @return boolean
     */ 
    public function clearCompiledTemplate($name = null, $time = null) 
    { 
        return $this->_smarty->clear_compiled_tpl($name, null, $time); 
    } 
 
    /**
     * Clear cached template
     *
     * @param string $name template name
     * @param string $time expiration time (second)
     * @return boolean
     */ 
    public function clearCachedTemplate($name = null, $time = null) 
    { 
        if ($this->_cacheMode === 0) 
        { 
            require_once 'Addax/View/Adapter/Exception.php'; 
            throw new Addax_View_Adapter_Exception(self::NOUSES_CACHE); 
        } 
 
        if ($name === null) 
        { 
            return $this->_smarty->clear_all_cache($time); 
        } else { 
            return $this->_smarty->clear_cache($name, null, null, $time); 
        } 
    } 
} 
?>
