<?php
/**
 * Zikula Application Framework
 *
 * @copyright (c) 2001, Zikula Development Team
 * @link http://www.zikula.org
 * @version $Id: pnadmin.php 28060 2010-01-07 08:31:22Z drak $
 * @license GNU/GPL - http://www.gnu.org/copyleft/gpl.html
 * @package Zikula_System_Modules
 * @subpackage Modules
 */

/**
 * Modules Module main admin function
 * @author Jim McDonald
 * @return string HTML output string
 */
function modules_admin_main()
{
    // security check
    if (!SecurityUtil::checkPermission('Modules::', '::', ACCESS_ADMIN)) {
        return LogUtil::registerPermissionError();
    }

    // Create output object - this object will store all of our output so that
    // we can return it easily when required
    $pnRender = & pnRender::getInstance('Modules', false);

    // Return the output that has been generated by this function
    return $pnRender->fetch('modules_admin_main.htm');
}

/**
 * Modules_admin_modify - modify a module
 * @author Jim McDonald
 * @return string HTML output string
 */
function modules_admin_modify()
{
    $id = (int) FormUtil::getPassedValue('id', null, 'GET');
    if (!is_numeric($id)) {
        return LogUtil::registerArgsError(pnModURL('Modules', 'admin', 'view'));
    }

    $obj = pnModGetInfo($id);
    if (!isset($id) || $obj == false) {
        return LogUtil::registerError(__('Error! No such module ID exists.'), 404, pnModURL('Modules', 'admin', 'modify', array(
            'id' => $id)));
    }

    if (!SecurityUtil::checkPermission('Modules::', "$obj[name]::$id", ACCESS_ADMIN)) {
        return LogUtil::registerPermissionError();
    }

    $restore = (bool)FormUtil::getPassedValue('restore', false, 'GET');
    if ($restore) {
        if ($obj['type'] == 3) {
            $pnversion = "system/$obj[directory]/pnversion.php";
            if (is_readable($pnversion)) {
                include $pnversion;
            } else {
                return LogUtil::registerError(__('Error! Unable to load version file for this module.'), 404, pnModURL('Modules', 'admin', 'modify', array(
                    'id' => $id)));
            }
        } else {
            ZLanguage::bindModuleDomain($obj['directory']);
            $pnversion = "modules/$obj[directory]/pnversion.php";
            $Version = "modules/$obj[directory]/Version.php";
            if (is_readable($pnversion)) {
                include $pnversion;
            } elseif (is_readable($Version)) {
                include $Version;
            } else {
                return LogUtil::registerError(__('Error! Unable to load version file for this module.'), 404, pnModURL('Modules', 'admin', 'modify', array(
                    'id' => $id)));
            }
        }

        // load defaults
        $name = (isset($modversion['name']) ? $modversion['name'] : '');
        $displayname = (isset($modversion['displayname']) ? $modversion['displayname'] : $name);
        $url = (isset($modversion['url']) ? $modversion['url'] : $displayname);
        $description = (isset($modversion['description']) ? $modversion['description'] : '');

        $obj = array(
            'id' => $id,
            'displayname' => $displayname,
            'url' => $url,
            'description' => $description);
    }

    $pnRender = & pnRender::getInstance('Modules', false);
    $pnRender->assign($obj);

    // Return the output that has been generated by this function
    return $pnRender->fetch('modules_admin_modify.htm');
}

/**
 * Modules_admin_update - update a module
 * @author Jim McDonald
 * @return string HTML output string
 * @param int 'id' module id
 * @param string 'newdisplayname' new display name of the module
 * @param string 'newdescription' new description of the module
 */
function modules_admin_update()
{
    // Get parameters
    $id = (int) FormUtil::getPassedValue('id', null, 'POST');
    $newdisplayname = FormUtil::getPassedValue('newdisplayname', null, 'POST');
    $newdescription = FormUtil::getPassedValue('newdescription', null, 'POST');
    $newurl = FormUtil::getPassedValue('newurl', null, 'POST');

    if (!SecurityUtil::confirmAuthKey()) {
        return LogUtil::registerAuthidError(pnModURL('Modules', 'admin', 'view'));
    }

    // Pass to API
    if (pnModAPIFunc('Modules', 'admin', 'update', array(
        'id' => $id,
        'displayname' => $newdisplayname,
        'description' => $newdescription,
        'url' => $newurl))) {
        // Success
        LogUtil::registerStatus(__('Done! Saved module information.'));
    } else {
        return pnRedirect(pnModURL('Modules', 'admin', 'modify', array('id' => $id)));
    }

    return pnRedirect(pnModURL('Modules', 'admin', 'view'));
}

/**
 * Modules_admin_update hooks
 * update hooks for a module
 * @author Mark West
 * @param int 'id' module id
 * @return bool true if successful, false otherwise
 */
function modules_admin_updatehooks()
{
    // Get parameters
    $id = (int) FormUtil::getPassedValue('id', null, 'REQUEST');

    if (!SecurityUtil::confirmAuthKey()) {
        return LogUtil::registerAuthidError(pnModURL('Modules', 'admin', 'view'));
    }

    // Pass to API
    if (pnModAPIFunc('Modules', 'admin', 'updatehooks', array(
        'id' => $id))) {
        // Success
        LogUtil::registerStatus(__('Done! Saved module information.'));
    }

    return pnRedirect(pnModURL('Modules', 'admin', 'view'));
}

/**
 * Modules_admin_update hooks, extended version
 * update hooks for a module
 * @author Mark West
 * @author Frank Schummertz
 * @param int 'id' module id
 * @return bool true if successful, false otherwise
 */
function modules_admin_extendedupdatehooks()
{
    // Get parameters
    $id = (int) FormUtil::getPassedValue('id', null, 'REQUEST');

    if (!SecurityUtil::confirmAuthKey()) {
        return LogUtil::registerAuthidError(pnModURL('Modules', 'admin', 'view'));
    }

    // Pass to API
    if (pnModAPIFunc('Modules', 'admin', 'extendedupdatehooks', array(
        'id' => $id))) {
        // Success
        LogUtil::registerStatus(__('Done! Saved module information.'));
    }

    return pnRedirect(pnModURL('Modules', 'admin', 'view'));
}

/**
 * Modules_admin_view - list modules and current settings
 * @author Jim McDonald
 * @return string HTML output string
 */
function modules_admin_view()
{
    // Security check
    if (!SecurityUtil::checkPermission('Modules::', '::', ACCESS_ADMIN)) {
        return LogUtil::registerPermissionError();
    }

    // Get parameters from whatever input we need.
    $startnum = (int) FormUtil::getPassedValue('startnum', null, 'REQUEST');
    $letter = FormUtil::getPassedValue('letter', null, 'REQUEST');
    $state = FormUtil::getPassedValue('state', (!strstr(pnServerGetVar('HTTP_REFERER'), 'module=Modules')) ? null : SessionUtil::getVar('state', null), 'REQUEST');
    $sort = FormUtil::getPassedValue('sort', (!strstr(pnServerGetVar('HTTP_REFERER'), 'module=Modules')) ? null : SessionUtil::getVar('sort', null), 'REQUEST');

    // do some clean up
    SessionUtil::delVar('interactive_init');
    SessionUtil::delVar('interactive_remove');
    SessionUtil::delVar('interactive_upgrade');

    // Create output object
    $pnRender = & pnRender::getInstance('Modules', false);

    // assign the state filter
    $pnRender->assign('state', $state);

    // Get list of modules
    $mods = pnModAPIFunc('Modules', 'admin', 'list', array(
        'startnum' => $startnum,
        'letter' => $letter,
        'state' => $state,
        'numitems' => pnModGetVar('Modules', 'itemsperpage'),
        'sort' => $sort));

    // generate an auth key to use in urls
    $authid = SecurityUtil::generateAuthKey();

    $moduleinfo = array();
    if (!empty($mods)) {
        foreach ($mods as $mod) {
            // Add applicable actions
            $actions = array();

            if (SecurityUtil::checkPermission('Modules::', "$mod[name]::$mod[id]", ACCESS_ADMIN)) {
                switch ($mod['state']) {
                    case PNMODULE_STATE_ACTIVE:
                        $actions[] = array(
                            'url' => pnModURL('Modules', 'admin', 'deactivate', array(
                                'id' => $mod['id'],
                                'startnum' => $startnum,
                                'authid' => $authid,
                                'letter' => $letter,
                                'state' => $state)),
                            'image' => 'folder_grey.gif',
                            'title' => __('Deactivate'));
                        $actions[] = array(
                            'url' => pnModURL('Modules', 'admin', 'hooks', array(
                                'id' => $mod['id'])),
                            'image' => 'attach.gif',
                            'title' => __('Hook settings'));
                        break;
                    case PNMODULE_STATE_INACTIVE:
                        $actions[] = array(
                            'url' => pnModURL('Modules', 'admin', 'activate', array(
                                'id' => $mod['id'],
                                'startnum' => $startnum,
                                'authid' => $authid,
                                'letter' => $letter,
                                'state' => $state)),
                            'image' => 'folder_green.gif',
                            'title' => __('Activate'));
                        $actions[] = array(
                            'url' => pnModURL('Modules', 'admin', 'remove', array(
                                'id' => $mod['id'],
                                'startnum' => $startnum,
                                'authid' => $authid,
                                'letter' => $letter,
                                'state' => $state)),
                            'image' => '14_layer_deletelayer.gif',
                            'title' => __('Remove module'));
                        break;
                    case PNMODULE_STATE_MISSING:
                        $actions[] = array(
                            'url' => pnModURL('Modules', 'admin', 'remove', array(
                                'id' => $mod['id'],
                                'startnum' => $startnum,
                                'authid' => $authid,
                                'letter' => $letter,
                                'state' => $state)),
                            'image' => '14_layer_deletelayer.gif',
                            'title' => __('Remove module'));
                        break;
                    case PNMODULE_STATE_UPGRADED:
                        $actions[] = array(
                            'url' => pnModURL('Modules', 'admin', 'upgrade', array(
                                'id' => $mod['id'],
                                'startnum' => $startnum,
                                'authid' => $authid,
                                'letter' => $letter,
                                'state' => $state)),
                            'image' => 'agt_update-product.gif',
                            'title' => __('Upgrade'));
                        break;
                    case PNMODULE_STATE_INVALID:
                        $actions[] = array(
                            'url' => pnModURL('Modules', 'admin', 'remove', array(
                                'id' => $mod['id'],
                                'startnum' => $startnum,
                                'authid' => $authid,
                                'letter' => $letter,
                                'state' => $state)),
                            'image' => '14_layer_deletelayer.gif',
                            'title' => __('Remove module'));
                        break;
                    case PNMODULE_STATE_NOTALLOWED:
                        $actions[] = array(
                            'url' => pnModURL('Modules', 'admin', 'remove', array(
                                'id' => $mod['id'],
                                'startnum' => $startnum,
                                'authid' => $authid,
                                'letter' => $letter,
                                'state' => $state)),
                            'image' => '14_layer_deletelayer.gif',
                            'title' => __('Remove module'));
                        break;
                    case PNMODULE_STATE_UNINITIALISED:
                    default:
                        $actions[] = array(
                            'url' => pnModURL('Modules', 'admin', 'initialise', array(
                                'id' => $mod['id'],
                                'startnum' => $startnum,
                                'authid' => $authid,
                                'letter' => $letter,
                                'state' => $state)),
                            'image' => 'agt_update_misc.gif',
                            'title' => __('Install'));
                        if ($GLOBALS['PNConfig']['Multisites']['multi'] != 1 || $GLOBALS['PNConfig']['Multisites']['mainSiteURL'] == FormUtil::getPassedValue('siteDNS', null, 'GET')) {
                            $actions[] = array(
                                'url' => pnModURL('Modules', 'admin', 'remove', array(
                                    'id' => $mod['id'],
                                    'startnum' => $startnum,
                                    'authid' => $authid,
                                    'letter' => $letter,
                                    'state' => $state)),
                                'image' => '14_layer_deletelayer.gif',
                                'title' => __('Remove module'));
                        }
                        break;
                }
                // RNG: can't edit an invalid module
                if ($mod['state'] != PNMODULE_STATE_INVALID)
                    $actions[] = array(
                        'url' => pnModURL('Modules', 'admin', 'modify', array(
                            'id' => $mod['id'])),
                        'image' => 'xedit.gif',
                        'title' => __('Edit'));
            }

            // Translate state
            switch ($mod['state']) {
                case PNMODULE_STATE_INACTIVE:
                    $status = __('Inactive');
                    $statusimage = 'yellowled.gif';
                    break;
                case PNMODULE_STATE_ACTIVE:
                    $status = __('Active');
                    $statusimage = 'greenled.gif';
                    break;
                case PNMODULE_STATE_MISSING:
                    $status = __('Files missing');
                    $statusimage = 'redled.gif';
                    break;
                case PNMODULE_STATE_UPGRADED:
                    $status = __('New version uploaded');
                    $statusimage = 'redled.gif';
                    break;
                case PNMODULE_STATE_INVALID:
                    $status = __('Invalid structure');
                    $statusimage = 'redled.gif';
                    break;
                case PNMODULE_STATE_NOTALLOWED:
                    $status = __('Not allowed');
                    $statusimage = 'redled.gif';
                    break;
                case PNMODULE_STATE_UNINITIALISED:
                default:
                    $status = __('Not installed');
                    $statusimage = 'redled.gif';
                    break;
            }

            $moduleinfo[] = array(
                'modinfo' => $mod,
                'status' => $status,
                'statusimage' => $statusimage,
                'options' => $actions);

        }
    }

    $pnRender->assign('multi', $GLOBALS['PNConfig']['Multisites']['multi']);

    $pnRender->assign('modules', $moduleinfo);

    // Assign the values for the smarty plugin to produce a pager.
    $pnRender->assign('pager', array(
        'numitems' => pnModAPIFunc('Modules', 'admin', 'countitems', array(
            'letter' => $letter,
            'state' => $state)),
        'itemsperpage' => pnModGetVar('Modules', 'itemsperpage')));

    // Return the output that has been generated by this function
    return $pnRender->fetch('modules_admin_view.htm');
}

/**
 * Modules_admin_view - list modules and current settings
 *
 * This function is an alias to modules_admin_view as pnphpbb calls this
 * function name directly on the install
 *
 * @author Jim McDonald
 * @see modules_admin_view
 * @return string HTML output string
 */
function modules_admin_list()
{
    return modules_admin_view();
}

/**
 * Initialise a module
 * @author Jim McDonald
 * @param int 'id' module id
 * @return bool true
 */
function modules_admin_initialise()
{
    // Security and sanity checks
    if (!SecurityUtil::confirmAuthKey()) {
        return LogUtil::registerAuthidError(pnModURL('Modules', 'admin', 'view'));
    }

    // Get parameters from whatever input we need
    $id = (int) FormUtil::getPassedValue('id', 0);
    $objectid = (int) FormUtil::getPassedValue('objectid', 0);
    $confirmation = (bool) FormUtil::getPassedValue('confirmation', false);
    $startnum = (int) FormUtil::getPassedValue('startnum');
    $letter = FormUtil::getPassedValue('letter');
    $state = FormUtil::getPassedValue('state');
    if ($objectid) {
        $id = $objectid;
    }

    // assign any dependencies - filtering out non-active module dependents
    // when getting here without a valid id we are in interactive init mode and then
    // the dependencies checks have been done before already
    if ($id != 0) {
        $dependencies = pnModAPIFunc('Modules', 'admin', 'getdependencies', array(
            'modid' => $id));
        $modulenotfound = false;
        if (empty($confirmation) && $dependencies) {
            foreach ($dependencies as $key => $dependency) {
                $dependencies[$key]['insystem'] = true;
                $modinfo = pnModGetInfo(pnModGetIDFromName($dependency['modname']));
                if (pnModAvailable($dependency['modname'])) {
                    unset($dependencies[$key]);
                } elseif (!empty($modinfo)) {
                    $dependencies[$key] = array_merge($dependencies[$key], $modinfo);
                } else {
                    $dependencies[$key]['insystem'] = false;
                    $modulenotfound = true;
                }
            }

            // we have some dependencies so let's warn the user about these
            if (!empty($dependencies)) {
                $pnRender = & pnRender::getInstance('Modules', false);
                $pnRender->assign('id', $id);
                $pnRender->assign('dependencies', $dependencies);
                $pnRender->assign('modulenotfound', $modulenotfound);
                return $pnRender->fetch('modules_admin_initialise.htm');
            }
        } else {
            $dependencies = (array) FormUtil::getPassedValue('dependencies', array(), 'POST');
        }
    }

    $interactive_init = SessionUtil::getVar('interactive_init');
    $interactive_init = (empty($interactive_init)) ? false : true;
    if ($interactive_init == false) {
        SessionUtil::setVar('modules_id', $id);
        SessionUtil::setVar('modules_startnum', $startnum);
        SessionUtil::setVar('modules_letter', $letter);
        SessionUtil::setVar('modules_state', $state);
        $activate = false;
    } else {
        $id = SessionUtil::getVar('modules_id');
        $startnum = SessionUtil::getVar('modules_startnum');
        $letter = SessionUtil::getVar('modules_letter');
        $state = SessionUtil::getVar('modules_state');
        $activate = (bool) FormUtil::getPassedValue('activate');
    }

    if (empty($id) || !is_numeric($id)) {
        return LogUtil::registerError(__('Error! No module ID provided.'), 404, pnModURL('Modules', 'admin', 'view'));
    }

    // initialise and activate any dependencies
    if (isset($dependencies) && is_array($dependencies)) {
        foreach ($dependencies as $dependency) {
            if (!pnModAPIFunc('Modules', 'admin', 'initialise', array(
                'id' => $dependency))) {
                return pnRedirect(pnModURL('Modules', 'admin', 'view', array(
                    'startnum' => $startnum,
                    'letter' => $letter,
                    'state' => $state)));
            }
            if (!pnModAPIFunc('Modules', 'admin', 'setstate', array(
                'id' => $dependency,
                'state' => PNMODULE_STATE_ACTIVE))) {
                return pnRedirect(pnModURL('Modules', 'admin', 'view', array(
                    'startnum' => $startnum,
                    'letter' => $letter,
                    'state' => $state)));
            }
        }
    }

    // Now we've initialised the dependencies initialise the main module
    $res = pnModAPIFunc('Modules', 'admin', 'initialise', array(
        'id' => $id,
        'interactive_init' => $interactive_init));
    if (is_bool($res) && $res == true) {
        // Success
        SessionUtil::delVar('modules_id');
        SessionUtil::delVar('modules_startnum');
        SessionUtil::delVar('modules_letter');
        SessionUtil::delVar('modules_state');
        SessionUtil::delVar('interactive_init');
        LogUtil::registerStatus(__('Done! Installed module.'));

        if ($activate == true) {
            if (pnModAPIFunc('Modules', 'admin', 'setstate', array(
                'id' => $id,
                'state' => PNMODULE_STATE_ACTIVE))) {
                // Success
                LogUtil::registerStatus(__('Done! Activated module.'));
            }
        }
        return pnRedirect(pnModURL('Modules', 'admin', 'view', array(
            'startnum' => $startnum,
            'letter' => $letter,
            'state' => $state)));
    } elseif (is_bool($res)) {
        return pnRedirect(pnModURL('Modules', 'admin', 'view', array(
            'startnum' => $startnum,
            'letter' => $letter,
            'state' => $state)));
    } else {
        return $res;
    }
}

/**
 * Activate a module
 * @author Jim McDonald
 * @param int 'id' module id
 * @return bool true
 */
function modules_admin_activate()
{
    // Security and sanity checks
    if (!SecurityUtil::confirmAuthKey()) {
        return LogUtil::registerAuthidError(pnModURL('Modules', 'admin', 'view'));
    }

    $id = (int) FormUtil::getPassedValue('id', null, 'GET');
    $startnum = (int) FormUtil::getPassedValue('startnum', null, 'GET');
    $letter = FormUtil::getPassedValue('letter', null, 'GET');
    $state = FormUtil::getPassedValue('state', null, 'GET');
    if (empty($id) || !is_numeric($id)) {
        return LogUtil::registerError(__('Error! No module ID provided.'), 404, pnModURL('Modules', 'admin', 'view'));
    }

    $moduleinfo = pnModGetInfo($id);
    if ($moduleinfo['state'] == 6) {
        LogUtil::registerError(__('Error! Module not allowed.'));
        return pnRedirect(pnModURL('Modules', 'admin', 'view', array(
            'startnum' => $startnum,
            'letter' => $letter,
            'state' => $state)));
    }

    // Update state
    if (pnModAPIFunc('Modules', 'admin', 'setstate', array(
        'id' => $id,
        'state' => PNMODULE_STATE_ACTIVE))) {
        // Success
        LogUtil::registerStatus(__('Done! Activated module.'));
    }

    return pnRedirect(pnModURL('Modules', 'admin', 'view', array(
        'startnum' => $startnum,
        'letter' => $letter,
        'state' => $state)));
}

/**
 * Upgrade a module
 * @author Jim McDonald
 * @param int 'id' module id
 * @return bool true
 */
function modules_admin_upgrade()
{
    // Security and sanity checks
    if (!SecurityUtil::confirmAuthKey()) {
        return LogUtil::registerAuthidError(pnModURL('Modules', 'admin', 'view'));
    }

    $interactive_upgrade = SessionUtil::getVar('interactive_upgrade');
    $interactive_upgrade = (empty($interactive_upgrade)) ? false : true;
    if ($interactive_upgrade == false) {
        $id = (int) FormUtil::getPassedValue('id', null, 'GET');
        $startnum = (int) FormUtil::getPassedValue('startnum', null, 'GET');
        $letter = FormUtil::getPassedValue('letter', null, 'GET');
        $state = FormUtil::getPassedValue('state', null, 'GET');
        SessionUtil::setVar('modules_id', $id);
        SessionUtil::setVar('modules_startnum', $startnum);
        SessionUtil::setVar('modules_letter', $letter);
        SessionUtil::setVar('modules_state', $state);
        $activate = false;
    } else {
        $id = SessionUtil::getVar('modules_id');
        $startnum = SessionUtil::getVar('modules_startnum');
        $letter = SessionUtil::getVar('modules_letter');
        $state = SessionUtil::getVar('modules_state');
        $activate = (bool) FormUtil::getPassedValue('activate', null, 'POST');
    }
    if (empty($id) || !is_numeric($id)) {
        return LogUtil::registerError(__('Error! No module ID provided.'), 404, pnModURL('Modules', 'admin', 'view'));
    }

    // Upgrade module
    $res = pnModAPIFunc('Modules', 'admin', 'upgrade', array(
        'id' => $id,
        'interactive_upgrade' => $interactive_upgrade));
    if (is_bool($res) && $res == true) {
        // Success
        SessionUtil::delVar('modules_id');
        SessionUtil::delVar('modules_startnum');
        SessionUtil::delVar('modules_letter');
        SessionUtil::delVar('modules_state');
        SessionUtil::setVar('interactive_upgrade', false);
        LogUtil::registerStatus(__('New version uploaded'));
        if ($activate == true) {
            if (pnModAPIFunc('Modules', 'admin', 'setstate', array(
                'id' => $id,
                'state' => PNMODULE_STATE_ACTIVE))) {
                // Success
                LogUtil::registerStatus(__('Activated'));
            }
        }
        return pnRedirect(pnModURL('Modules', 'admin', 'view', array(
            'startnum' => $startnum,
            'letter' => $letter,
            'state' => $state)));
    } elseif (is_bool($res)) {
        return pnRedirect(pnModURL('Modules', 'admin', 'view', array(
            'startnum' => $startnum,
            'letter' => $letter,
            'state' => $state)));
    } else {
        return $res;
    }
}

/**
 * Deactivate a module
 * @author Jim McDonald
 * @param int 'id' module id
 * @return bool true
 */
function modules_admin_deactivate()
{
    // Security and sanity checks
    if (!SecurityUtil::confirmAuthKey()) {
        return LogUtil::registerAuthidError(pnModURL('Modules', 'admin', 'view'));
    }

    $id = (int) FormUtil::getPassedValue('id', null, 'GET');
    $startnum = (int) FormUtil::getPassedValue('startnum', null, 'GET');
    $letter = FormUtil::getPassedValue('letter', null, 'GET');
    $state = FormUtil::getPassedValue('state', null, 'GET');
    if (empty($id) || !is_numeric($id)) {
        return LogUtil::registerError(__('Error! No module ID provided.'), 404, pnModURL('Modules', 'admin', 'view'));
    }

    // check if the modules is the systems start module
    $modinfo = pnModGetInfo($id);
    if ($modinfo == false) {
        return LogUtil::registerError(__('Error! No such module ID exists.'), 404, pnModURL('Modules', 'admin', 'view'));
    }
    $startmod = pnConfigGetVar('startpage');
    if ($startmod == $modinfo['name']) {
        return LogUtil::registerError(__('Error! This module is currently set as the site\'s home page. You must choose another module for the home page before you can deactivate this one.'), null, pnModURL('Modules', 'admin', 'view'));
    }

    $coremodules = array(
        'Modules',
        'Permissions',
        'Groups',
        'Blocks',
        'ObjectData',
        'Users',
        'Theme',
        'Admin',
        'Settings',
        'Categories');
    if (in_array($modinfo['name'], $coremodules)) {
        return LogUtil::registerError(__('Error! You cannot deactivate this module. It is a mandatory core module, and is needed by the system.'), null, pnModURL('Modules', 'admin', 'view'));
    }

    // Update state
    if (pnModAPIFunc('Modules', 'admin', 'setstate', array(
        'id' => $id,
        'state' => PNMODULE_STATE_INACTIVE))) {
        // Success
        LogUtil::registerStatus(__('Done! Deactivated module.'));
    }

    return pnRedirect(pnModURL('Modules', 'admin', 'view', array(
        'startnum' => $startnum,
        'letter' => $letter,
        'state' => $state)));
}

/**
 * Remove a module
 * @author Jim McDonald
 * @param int 'id' module id
 * @return bool true if successful
 */
function modules_admin_remove()
{
    // Get parameters from whatever input we need
    $id = (int) FormUtil::getPassedValue('id', 0);
    $objectid = (int) FormUtil::getPassedValue('objectid', 0);
    $confirmation = (bool) FormUtil::getPassedValue('confirmation', false);
    $dependents = (array) FormUtil::getPassedValue('dependents');
    $startnum = (int) FormUtil::getPassedValue('startnum');
    $letter = FormUtil::getPassedValue('letter');
    $state = FormUtil::getPassedValue('state');
    if ($objectid) {
        $id = $objectid;
    }

    $interactive_remove = SessionUtil::getVar('interactive_remove');
    $interactive_remove = (empty($interactive_remove)) ? false : true;

    if ($interactive_remove == false) {
        SessionUtil::setVar('modules_id', $id);
        SessionUtil::setVar('modules_startnum', $startnum);
        SessionUtil::setVar('modules_letter', $letter);
        SessionUtil::setVar('modules_state', $state);
    } else {
        $id = SessionUtil::getVar('modules_id');
        $startnum = SessionUtil::getVar('modules_startnum');
        $letter = SessionUtil::getVar('modules_letter');
        $state = SessionUtil::getVar('modules_state');
        $confirmation = 1;
    }

    if (empty($id) || !is_numeric($id) || !pnModGetInfo($id)) {
        return LogUtil::registerError(__('Error! No module ID provided.'), 404, pnModURL('Modules', 'admin', 'view'));
    }

    // Check for confirmation.
    if (empty($confirmation)) {
        // No confirmation yet
        // Create output object
        $pnRender = & pnRender::getInstance('Modules', false);

        // Add a hidden field for the item ID to the output
        $pnRender->assign('id', $id);

        // assign any dependencies - filtering out non-active module dependents
        $dependents = pnModAPIFunc('Modules', 'admin', 'getdependents', array(
            'modid' => $id));
        foreach ($dependents as $key => $dependent) {
            $modinfo = pnModGetInfo($dependent['modid']);
            if (!pnModAvailable($modinfo['name'])) {
                unset($dependents[$key]);
            } else {
                $dependents[$key] = array_merge($dependents[$key], $modinfo);
            }
        }
        // check the blocks module for existing blocks
        $blocks = pnModAPIFunc('Blocks', 'user', 'getall', array(
            'modid' => $id));
        $pnRender->assign('hasBlocks', !empty($blocks));

        $pnRender->assign('dependents', $dependents);
        $pnRender->assign('startnum', $startnum);
        $pnRender->assign('letter', $letter);
        $pnRender->assign('state', $state);

        // Return the output that has been generated by this function
        return $pnRender->fetch('modules_admin_remove.htm');
    }

    // If we get here it means that the user has confirmed the action


    // Security and sanity checks
    if (!SecurityUtil::confirmAuthKey()) {
        return LogUtil::registerAuthidError(pnModURL('Modules', 'admin', 'view'));
    }

    // check the blocks module for existing blocks
    $blocks = pnModAPIFunc('Blocks', 'user', 'getall', array(
        'modid' => $id));
    if (!empty($blocks)) {
        return LogUtil::registerError(__('Error! This module cannot be removed at the moment because there are active blocks that interact with it. You must deactivate those blocks first before you can deactivate this module.'), null, pnModURL('Modules', 'admin', 'view'));
    }

    // remove dependent modules


    foreach ($dependents as $dependent) {
        if (!pnModAPIFunc('Modules', 'admin', 'remove', array(
            'id' => $dependent))) {
            return pnRedirect(pnModURL('Modules', 'admin', 'view', array(
                'startnum' => $startnum,
                'letter' => $letter,
                'state' => $state)));
        }
    }

    // Now we've removed dependents remove the main module
    $res = pnModAPIFunc('Modules', 'admin', 'remove', array(
        'id' => $id,
        'interactive_remove' => $interactive_remove));
    if (is_bool($res) && $res == true) {
        // Success
        SessionUtil::delVar('modules_id');
        SessionUtil::delVar('modules_startnum');
        SessionUtil::delVar('modules_letter');
        SessionUtil::delVar('modules_state');
        SessionUtil::delVar('interactive_remove');
        LogUtil::registerStatus(__('Done! De-installed module.'));
        return pnRedirect(pnModURL('Modules', 'admin', 'view', array(
            'startnum' => $startnum,
            'letter' => $letter,
            'state' => $state)));
    } elseif (is_bool($res)) {
        return pnRedirect(pnModURL('Modules', 'admin', 'view', array(
            'startnum' => $startnum,
            'letter' => $letter,
            'state' => $state)));
    } else {
        return $res;
    }
}

/**
 * Re-generate modules list
 * @author Jim McDonald
 * @author  Jörg Napp
 */
function modules_admin_regenerate()
{
    // Security check
    if (!SecurityUtil::confirmAuthKey()) {
        return LogUtil::registerAuthidError(pnModURL('Modules', 'admin', 'view'));
    }

    if ($GLOBALS['PNConfig']['Multisites']['multi'] == 1) {
        // only the main site can regenerate the modules list
        if ($GLOBALS['PNConfig']['Multisites']['mainSiteURL'] != FormUtil::getPassedValue('siteDNS', null, 'GET')) {
            //return true but any action has been made
            return pnRedirect(pnModURL('Modules', 'admin', 'view', array(
                'startnum' => $startnum,
                'letter' => $letter,
                'state' => $state)));
        }
    }

    $defaults = (int) FormUtil::getPassedValue('defaults', false, 'GET');
    $filemodules = pnModAPIFunc('Modules', 'admin', 'getfilemodules');
    $inconsistencies = pnModAPIFunc('Modules', 'admin', 'checkconsistency', array(
        'filemodules' => $filemodules));

    if (empty($inconsistencies['errors_modulenames']) && empty($inconsistencies['errors_displaynames'])) {
        // No inconsistencies, so we can regenerate modules
        if (pnModAPIFunc('Modules', 'admin', 'regenerate', array(
            'filemodules' => $filemodules, 'defaults' => $defaults))) {
            // Success
            LogUtil::registerStatus(__('Done! Regenerated modules list from file system.'));
        }

        // get variables to pass back to pager
        $startnum = (int) FormUtil::getPassedValue('startnum', null, 'GET');
        $letter = FormUtil::getPassedValue('letter', null, 'GET');
        $state = FormUtil::getPassedValue('state', null, 'GET');

        return pnRedirect(pnModURL('Modules', 'admin', 'view', array(
            'startnum' => $startnum,
            'letter' => $letter,
            'state' => $state)));
    }

    // Create output object
    $pnRender = & pnRender::getInstance('Modules', false);

    $pnRender->assign('errors_modulenames', $inconsistencies['errors_modulenames']);
    $pnRender->assign('errors_displaynames', $inconsistencies['errors_displaynames']);
    $pnRender->assign('returnurl', $returnurl);

    return $pnRender->fetch('modules_admin_regenerate_errors.htm');

}

/**
 * display available hooks
 * @author Mark West
 * @param int 'id' module id
 * @return string HTML output string
 */
function modules_admin_hooks()
{
    // get our input
    $id = (int) FormUtil::getPassedValue('id', null, 'GET');

    // check the input
    if (!is_numeric($id)) {
        return LogUtil::registerArgsError(pnModURL('Modules', 'admin', 'view'));
    }

    // get the modules information
    $modinfo = pnModGetInfo($id);
    if ($modinfo == false) {
        return LogUtil::registerError(__('Error! No such module ID exists.'), 404, pnModURL('Modules', 'admin', 'view'));
    }

    if (!SecurityUtil::checkPermission('Modules::', "$modinfo[name]::$id", ACCESS_ADMIN)) {
        return LogUtil::registerPermissionError();
    }

    // Create output object - this object will store all of our output so that
    // we can return it easily when required
    $pnRender = & pnRender::getInstance('Modules', false);

    // assign the modinfo to the template
    $pnRender->assign('modinfo', $modinfo);

    // add module id to form
    $pnRender->assign('id', $id);

    $hooks = pnModAPIFunc('Modules', 'admin', 'getmoduleshooks', array(
        'modid' => $id));

    $pnRender->assign('hooks', $hooks);

    // Return the output that has been generated by this function
    return $pnRender->fetch('modules_admin_hooks.htm');

}

/**
 * display available hooks, extended version
 * @author Mark West
 * @author Frank Schummertz
 * @param int 'id' module id
 * @return string HTML output string
 */
function modules_admin_extendedhooks()
{
    // get our input
    $id = (int) FormUtil::getPassedValue('id', null, 'GET');

    // check the input
    if (!is_numeric($id)) {
        return LogUtil::registerArgsError(pnModURL('Modules', 'admin', 'view'));
    }

    // get the modules information
    $modinfo = pnModGetInfo($id);
    if ($modinfo == false) {
        return LogUtil::registerError(__('Error! No such module ID exists.'), 404, pnModURL('Modules', 'admin', 'view'));
    }

    if (!SecurityUtil::checkPermission('Modules::', "$modinfo[name]::$id", ACCESS_ADMIN)) {
        return LogUtil::registerPermissionError();
    }

    // Create output object - this object will store all of our output so that
    // we can return it easily when required
    $pnRender = & pnRender::getInstance('Modules', false);

    // assign the modinfo to the template
    $pnRender->assign('modinfo', $modinfo);

    // add module id to form
    $pnRender->assign('id', $id);

    $grouped_hooks = pnModAPIFunc('Modules', 'admin', 'getextendedmoduleshooks', array(
        'modid' => $id));
    $pnRender->assign('grouped_hooks', $grouped_hooks);

    // Return the output that has been generated by this function
    return $pnRender->fetch('modules_admin_extendedhooks.htm');

}

/**
 * This is a standard function to modify the configuration parameters of the
 * module
 * @author Mark West
 * @since 1.14
 * @return string HTML string
 */
function modules_admin_modifyconfig()
{
    // Security check
    if (!SecurityUtil::checkPermission('Modules::', '::', ACCESS_ADMIN)) {
        return LogUtil::registerPermissionError();
    }

    // Create output object
    $pnRender = & pnRender::getInstance('Modules', false);

    // assign all the module vars
    $pnRender->assign(pnModGetVar('Modules'));

    // load support for legacy modules
    $pnRender->assign('loadlegacy', pnConfigGetVar('loadlegacy'));

    // Return the output that has been generated by this function
    return $pnRender->fetch('modules_admin_modifyconfig.htm');
}

/**
 * This is a standard function to update the configuration parameters of the
 * module given the information passed back by the modification form
 * @author Mark West
 * @since 1.14
 * @return bool true
 */
function modules_admin_updateconfig()
{
    // Security check
    if (!SecurityUtil::checkPermission('Modules::', '::', ACCESS_ADMIN)) {
        return LogUtil::registerPermissionError();
    }

    // Confirm authorisation code.
    if (!SecurityUtil::confirmAuthKey()) {
        return LogUtil::registerAuthidError(pnModURL('Modules', 'admin', 'view'));
    }

    // Update module variables.
    $itemsperpage = (int) FormUtil::getPassedValue('itemsperpage', 25, 'POST');
    if (!is_integer($itemsperpage) || $itemsperpage < 1) {
        LogUtil::registerError(__("Warning! The 'Items per page' setting must be a positive integer. The value you entered was corrected."));
        $itemsperpage = (int) $itemsperpage;
        if ($itemsperpage < 1) {
            $itemsperpage = 25;
        }
    }
    pnModSetVar('Modules', 'itemsperpage', $itemsperpage);

    $loadlegacy = (bool) FormUtil::getPassedValue('loadlegacy', false, 'POST');
    pnConfigSetVar('loadlegacy', $loadlegacy);

    // the module configuration has been updated successfuly
    LogUtil::registerStatus(__('Done! Saved module configuration.'));

    // Let any other modules know that the modules configuration has been updated
    pnModCallHooks('module', 'updateconfig', 'Modules', array(
        'module' => 'Modules'));

    // This function generated no output, and so now it is complete we redirect
    // the user to an appropriate page for them to carry on their work
    return pnRedirect(pnModURL('Modules', 'admin', 'view'));
}


